<?php

namespace App\Exceptions;

use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Validation\ValidationException;
use Laravel\Lumen\Exceptions\Handler as ExceptionHandler;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Throwable;


use App\Http\Helpers\ResponseBuilder;
use App\Modules\Shared\Exceptions\ActionRequestException;
use App\Modules\Shared\Exceptions\BasicRequestException;
use App\Modules\Shared\Exceptions\InvalidRequestException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\MethodNotAllowedHttpException;

class Handler extends ExceptionHandler
{
    /**
     * A list of the exception types that should not be reported.
     *
     * @var array
     */
    protected $dontReport = [
        AuthorizationException::class,
        HttpException::class,
        ModelNotFoundException::class,
        ValidationException::class,
    ];

    /**
     * Report or log an exception.
     *
     * This is a great spot to send exceptions to Sentry, Bugsnag, etc.
     *
     * @param  \Throwable  $exception
     * @return void
     *
     * @throws \Exception
     */
    public function report(Throwable $exception)
    {
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Throwable  $exception
     * @return \Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     *
     * @throws \Throwable
     */
        
    public function render($request, Throwable $exception)
    {
        if ($exception instanceof NotFoundHttpException) {
            $response["Response_Status"]           = 404;
            $response["Response_Code"]             = 404;
            $response["Response_Domain"]           = "global";
            $response["Response_Message"]          = "Endpoint not found";
            $response["Response_Data"]             = null;
            $response["Response_Error_Message"]    = "Not found";
            $response["Response_Error_Reason"]     = "Endpoint not found";
            return ResponseBuilder::Response($response);
        } elseif ($exception instanceof MethodNotAllowedHttpException) {
            $response["Response_Status"]           = 405;
            $response["Response_Code"]             = 405;
            $response["Response_Domain"]           = "global";
            $response["Response_Message"]          = "Method not allowed";
            $response["Response_Data"]             = null;
            $response["Response_Error_Message"]    = "Method not allowed";
            $response["Response_Error_Reason"] =
                "The HTTP verb is not supported by the URL endpoint used in the request";
            return ResponseBuilder::Response($response);
        } elseif ($exception instanceof InvalidRequestException) {
            $response = $exception->getResponse();
            return ResponseBuilder::Response($response);
        } elseif ($exception instanceof ActionRequestException) {
            $response = [];
            $response["Response_Status"]            = $exception->getResponse()->Response_Status;
            $response["Response_Code"]              = $exception->getResponse()->Response_Code;
            $response["Response_Domain"]            = '';
            $response["Response_Message"]           = $exception->getResponse()->Response_Message;
            $response["Response_Data"]              = null;
            $response["Response_Error_Message"]     = $exception->getResponse()->Response_Message;
            $response["Response_Error_Reason"]      = $exception->getResponse()->Response_Reason;
            return ResponseBuilder::Response($response);
        } elseif ($exception instanceof BasicRequestException) {
            $response = [];
            $response["Response_Status"]            = $exception->getStatus();
            $response["Response_Code"]              = $exception->getCode();
            $response["Response_Domain"]            = '';
            $response["Response_Message"]           = $exception->getMessage();
            $response["Response_Data"]              = null;
            $response["Response_Error_Message"]     = $exception->getMessage();
            $response["Response_Error_Reason"]      = $exception->getMessage();
            return ResponseBuilder::Response($response);
        } else {
            $code = env('PRODUCTION', '0') == '0' ? $exception->getCode() : '500';
            $message = env('PRODUCTION', '0') == '0' ? $exception->getMessage() : 'SERVER ERROR';
            $trace = env('PRODUCTION', '0') == '0' ? $exception->getTraceAsString() : 'SERVER ERROR';
            $response = [];
            $response["Response_Status"]            = 500;
            $response["Response_Code"]              = $code;
            $response["Response_Domain"]            = '';
            $response["Response_Message"]           = $message;
            $response["Response_Data"]              = null;
            $response["Response_Error_Message"]     = $message;
            $response["Response_Error_Reason"]      = $trace;
            return ResponseBuilder::Response($response);
        }
    }
}
