<?php

namespace App\Providers\ExcelReport;

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Style;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class ExcelReport {

    public function exportToExcel($data, $metadata, $pageList) {

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $this->render (
            $sheet,
            $data,
            $metadata,
            $pageList
        );
        
        foreach(range('A','AA') as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }
        $writer = IOFactory::createWriter($spreadsheet, "Xlsx");
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $metadata->title . '.xlsx"');
        $writer->save("php://output");
    }

    public function render (Worksheet $sheet, $data, $metadata, $pageList) {
        
        $currentRow = 0;
        foreach ($data as $element) {

            foreach ($pageList as $page) {
                $this->renderAllInOnePage ($sheet, $page, $element, $currentRow, $metadata);
            }
            
            $currentRow += $metadata->rowLength ? $metadata->rowLength : ($metadata->evaluateRowLength ? $metadata->evaluateRowLength() : 1);
        }
        
    }

    public function renderAllInOnePage (Worksheet $sheet, $page, $element, $initialRow, $metadata) {
        $letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        foreach ($page->run($element, $metadata) as $ind => $row) {
            $column = 0;
            $currentRow = ($initialRow + $ind + 1);
            foreach ($row as $orders) {
                foreach ($orders as $order) {
                    switch ($order['action']) {
                        case 'print': $column = $this->print($sheet, $order, $column, $currentRow); break;
                        
                        case '': $column = $this->pass($sheet, $order, $column, $currentRow); break;
                        case 'merge': $column = $this->merge($sheet, $order, $column, $currentRow); break;
                        case 'mergeborder': $column = $this->mergeborder($sheet, $order, $column, $currentRow); break;
                        case 'vmerge': $column = $this->vmerge($sheet, $order, $column, $currentRow); break;
                        case 'advance': $column = $this->advance($sheet, $order, $column, $currentRow); break;
                        case 'width': $column = $this->width($sheet, $order, $column, $currentRow); break;
                        case 'border': $column = $this->border($sheet, $order, $column, $currentRow); break;
                        case 'bold': $column = $this->bold($sheet, $order, $column, $currentRow); break;
                        case 'center': $column = $this->center($sheet, $order, $column, $currentRow); break;
                        case 'top': $column = $this->top ($sheet, $order, $column, $currentRow); break;
                        case 'left': $column = $this->left($sheet, $order, $column, $currentRow); break;
                        case 'wrap': $column = $this->wrap ($sheet, $order, $column, $currentRow); break;
                        case 'size': $column = $this->size ($sheet, $order, $column, $currentRow); break;
                        case 'width-auto': $column = $this->autosize ($sheet, $order, $column, $currentRow); break;
                    }
                }
                $column++;
            }
            
        }
    }

    public const letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    function columnLetter($c){
        $c = intval($c);
        //if ($c <= 0) return 'qw';
        if($c<26){
            $letter = self::letters[$c];
        }else{
            $letter = 'A'.self::letters[$c-26];
        }
        return $letter;
    }

    protected function print($sheet, $order, $column, $currentRow) {
        $range1 = $this->columnLetter($column) . $currentRow;
        $sheet->setCellValue("$range1", $order['text']);
        return $column;
    }
    protected function pass ($sheet, $order, $column, $currentRow) { return $column; }
    
    protected function mergeborder ($sheet, $order, $column, $currentRow) {
        $range1 = $this->columnLetter($column) . $currentRow;
        $range2 = $this->columnLetter($column + $order['lines']) . $currentRow;
        $sheet->mergeCells("$range1:$range2");
        for($ini=$column;$ini<($column + $order['lines']);$ini++){
            $sheet->getStyle($this->columnLetter($ini) . $currentRow)
            ->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);
        }
        $column += $order['lines'];
        return $column;
    }
    protected function merge ($sheet, $order, $column, $currentRow) {
        $range1 = $this->columnLetter($column) . $currentRow;
        $range2 = self::letters[$column + $order['lines']] . $currentRow;
        $sheet->mergeCells("$range1:$range2");
        $column += $order['lines'];
        return $column;
    }
    protected function vmerge ($sheet, $order, $column, $currentRow) {
        $range1 = self::letters[$column] . $currentRow;
        $range2 = self::letters[$column] . ($currentRow - (-$order['lines']));
        $sheet->mergeCells("$range1:$range2");
        return $column;
    }
    protected function size ($sheet, $order, $column, $currentRow) {
        
        $sheet->getStyle($this->columnLetter($column) . $currentRow)->getFont()->setSize($order['lines']);
        return $column;
    }
    protected function advance ($sheet, $order, $column, $currentRow) {
        $column += $order['lines'];
        return $column;
    }

    protected function width ($sheet, $order, $column, $currentRow) {
        $sheet->getColumnDimensionByColumn($column + 1)->setWidth($order['width']);
        return $column;
    }
    protected function autosize ($sheet, $order, $column, $currentRow) {
        foreach(range('A','AA') as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }
        return $column;
    }
    protected function border ($sheet, $order, $column, $currentRow) {
        $sheet->getStyle($this->columnLetter($column) . $currentRow)
            ->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);
        return $column;
    }

    protected function bold ($sheet, $order, $column, $currentRow) {
        $sheet->getStyle($this->columnLetter($column) . $currentRow)->getFont()->setBold(true); 
        return $column;
    }

    protected function center ($sheet, $order, $column, $currentRow) {
        $sheet ->getStyle($this->columnLetter($column) . $currentRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
        return $column;
    }

    protected function top ($sheet, $order, $column, $currentRow) {
        $sheet->getStyle($this->columnLetter($column) . $currentRow)->getAlignment()->setVertical(Alignment::VERTICAL_TOP);
        return $column;
    }

    protected function left ($sheet, $order, $column, $currentRow) {
        $sheet->getStyle($this->columnLetter($column) . $currentRow)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_LEFT);
        return $column;
    }

    protected function wrap ($sheet, $order, $column, $currentRow) {
        $sheet->getStyle($this->columnLetter($column) . $currentRow)->getAlignment()->setWrapText(true);
        return $column;
    }

}